/*
 * Decompiled with CFR 0.152.
 */
package edu.uci.ics.jung.layout.algorithms;

import com.google.common.base.Preconditions;
import com.google.common.cache.CacheBuilder;
import com.google.common.cache.CacheLoader;
import com.google.common.cache.LoadingCache;
import com.google.common.graph.EndpointPair;
import com.google.common.graph.Graph;
import edu.uci.ics.jung.algorithms.util.IterativeContext;
import edu.uci.ics.jung.layout.algorithms.AbstractIterativeLayoutAlgorithm;
import edu.uci.ics.jung.layout.model.LayoutModel;
import edu.uci.ics.jung.layout.model.Point;
import java.util.ConcurrentModificationException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class FRLayoutAlgorithm<N>
extends AbstractIterativeLayoutAlgorithm<N>
implements IterativeContext {
    private static final Logger log = LoggerFactory.getLogger(FRLayoutAlgorithm.class);
    private double forceConstant;
    private double temperature;
    private int currentIteration;
    private int mMaxIterations = 700;
    protected LoadingCache<N, Point> frNodeData = CacheBuilder.newBuilder().build(new CacheLoader<N, Point>(){

        public Point load(N node) {
            return Point.ORIGIN;
        }
    });
    private double attraction_multiplier = 0.75;
    private double attraction_constant;
    private double repulsion_multiplier = 0.75;
    protected double repulsion_constant;
    private double max_dimension;
    private boolean initialized = false;
    protected double EPSILON = 1.0E-6;

    @Override
    public void visit(LayoutModel<N> layoutModel) {
        if (log.isTraceEnabled()) {
            log.trace("visiting " + layoutModel);
        }
        super.visit(layoutModel);
        this.max_dimension = Math.max(layoutModel.getWidth(), layoutModel.getHeight());
        this.initialize();
    }

    public void setAttractionMultiplier(double attraction) {
        this.attraction_multiplier = attraction;
    }

    public void setRepulsionMultiplier(double repulsion) {
        this.repulsion_multiplier = repulsion;
    }

    public void initialize() {
        this.doInit();
    }

    private void doInit() {
        Graph graph = this.layoutModel.getGraph();
        if (graph != null && graph.nodes().size() > 0) {
            this.currentIteration = 0;
            this.temperature = this.layoutModel.getWidth() / 10;
            this.forceConstant = Math.sqrt(this.layoutModel.getHeight() * this.layoutModel.getWidth() / graph.nodes().size());
            this.attraction_constant = this.attraction_multiplier * this.forceConstant;
            this.repulsion_constant = this.repulsion_multiplier * this.forceConstant;
            this.initialized = true;
        }
    }

    @Override
    public synchronized void step() {
        if (!this.initialized) {
            this.doInit();
        }
        Graph graph = this.layoutModel.getGraph();
        ++this.currentIteration;
        while (true) {
            try {
                for (Object node1 : graph.nodes()) {
                    this.calcRepulsion(node1);
                }
            }
            catch (ConcurrentModificationException concurrentModificationException) {
                continue;
            }
            break;
        }
        while (true) {
            try {
                for (EndpointPair endpoints : graph.edges()) {
                    this.calcAttraction(endpoints);
                }
            }
            catch (ConcurrentModificationException concurrentModificationException) {
                continue;
            }
            break;
        }
        while (true) {
            try {
                for (Object node : graph.nodes()) {
                    if (this.layoutModel.isLocked(node)) continue;
                    this.calcPositions(node);
                }
            }
            catch (ConcurrentModificationException concurrentModificationException) {
                continue;
            }
            break;
        }
        this.cool();
    }

    protected synchronized void calcPositions(N node) {
        Point fvd = this.getFRData(node);
        if (fvd == null) {
            return;
        }
        Point xyd = (Point)this.layoutModel.apply(node);
        double deltaLength = Math.max(this.EPSILON, fvd.length());
        double positionX = xyd.x;
        double positionY = xyd.y;
        double newXDisp = fvd.x / deltaLength * Math.min(deltaLength, this.temperature);
        double newYDisp = fvd.y / deltaLength * Math.min(deltaLength, this.temperature);
        positionX += newXDisp;
        positionY += newYDisp;
        double borderWidth = (double)this.layoutModel.getWidth() / 50.0;
        if (positionX < borderWidth) {
            positionX = borderWidth + this.random.nextDouble() * borderWidth * 2.0;
        } else if (positionX > (double)this.layoutModel.getWidth() - borderWidth * 2.0) {
            positionX = (double)this.layoutModel.getWidth() - borderWidth - this.random.nextDouble() * borderWidth * 2.0;
        }
        if (positionY < borderWidth) {
            positionY = borderWidth + this.random.nextDouble() * borderWidth * 2.0;
        } else if (positionY > (double)this.layoutModel.getWidth() - borderWidth * 2.0) {
            positionY = (double)this.layoutModel.getWidth() - borderWidth - this.random.nextDouble() * borderWidth * 2.0;
        }
        this.layoutModel.set(node, positionX, positionY);
    }

    protected void calcAttraction(EndpointPair<N> endpoints) {
        Object node1 = endpoints.nodeU();
        Object node2 = endpoints.nodeV();
        boolean v1_locked = this.layoutModel.isLocked(node1);
        boolean v2_locked = this.layoutModel.isLocked(node2);
        if (v1_locked && v2_locked) {
            return;
        }
        Point p1 = (Point)this.layoutModel.apply(node1);
        Point p2 = (Point)this.layoutModel.apply(node2);
        if (p1 == null || p2 == null) {
            return;
        }
        double xDelta = p1.x - p2.x;
        double yDelta = p1.y - p2.y;
        double deltaLength = Math.max(this.EPSILON, Math.sqrt(xDelta * xDelta + yDelta * yDelta));
        double force = deltaLength * deltaLength / this.attraction_constant;
        Preconditions.checkState((!Double.isNaN(force) ? 1 : 0) != 0, (Object)"Unexpected mathematical result in FRLayout:calcPositions [force]");
        double dx = xDelta / deltaLength * force;
        double dy = yDelta / deltaLength * force;
        if (!v1_locked) {
            Point fvd1 = this.getFRData(node1);
            this.frNodeData.put(node1, (Object)fvd1.add(-dx, -dy));
        }
        if (!v2_locked) {
            Point fvd2 = this.getFRData(node2);
            this.frNodeData.put(node2, (Object)fvd2.add(dx, dy));
        }
    }

    protected void calcRepulsion(N node1) {
        Point fvd1 = this.getFRData(node1);
        if (fvd1 == null) {
            return;
        }
        this.frNodeData.put(node1, (Object)Point.ORIGIN);
        try {
            for (Object node2 : this.layoutModel.getGraph().nodes()) {
                if (node1 == node2) continue;
                fvd1 = this.getFRData(node1);
                Point p1 = (Point)this.layoutModel.apply(node1);
                Point p2 = (Point)this.layoutModel.apply(node2);
                if (p1 == null || p2 == null) continue;
                double xDelta = p1.x - p2.x;
                double yDelta = p1.y - p2.y;
                double deltaLength = Math.max(this.EPSILON, Math.sqrt(xDelta * xDelta + yDelta * yDelta));
                double force = this.repulsion_constant * this.repulsion_constant / deltaLength;
                if (Double.isNaN(force)) {
                    throw new RuntimeException("Unexpected mathematical result in FRLayout:calcPositions [repulsion]");
                }
                fvd1 = fvd1.add(xDelta / deltaLength * force, yDelta / deltaLength * force);
                this.frNodeData.put(node1, (Object)fvd1);
            }
        }
        catch (ConcurrentModificationException cme) {
            this.calcRepulsion(node1);
        }
    }

    private void cool() {
        this.temperature *= 1.0 - (double)this.currentIteration / (double)this.mMaxIterations;
    }

    public void setMaxIterations(int maxIterations) {
        this.mMaxIterations = maxIterations;
    }

    protected Point getFRData(N node) {
        return (Point)this.frNodeData.getUnchecked(node);
    }

    @Override
    public boolean done() {
        return this.currentIteration > this.mMaxIterations || this.temperature < 1.0 / this.max_dimension;
    }
}

